-- SETTINGS BY MA'RUF ID YOUTUBE CHANNEL --

local TableUtil = (...)
local customization = {}

-----------------------------------------------------------------------------------------------------
local BND_MATCH_SETTINGS_LIST_DATA = "bnd_match_settings_list_data"
-----------------------------------------------------------------------------------------------------

local bndCameraOptions = "bnd_camera_options"
local bndAutoSwitchOptions = "bnd_auto_switch_options"
local bndAutoSwitchMoveAssistOptions = "bnd_auto_switch_move_assist_options"
local bndMoveAssistanceOptions = "bnd_move_assistance_options"
local bndLargeButtonsOptions = "bnd_large_buttons_options"
local bndLargeDpadOptions = "bnd_large_dpad_options"
local bndDifficultyOptions = "bnd_difficulty_options"
local bndHalfLengthOptions = "bnd_half_length_options"

local actChangeCameraOption = "act_change_camera_option"
local actChangeAutoSwitchOption = "act_change_auto_switch_option"
local actChangeAutoSwitchMoveAssistOption = "act_change_auto_switch_move_assist_option"
local actChangeMoveAssistanceOption = "act_change_move_assistance_option"
local actChangeLargeButtonsOption = "act_change_large_buttons_option"
local actChangeLargeDpadOption = "act_change_large_dpad_option"
local actChangeDifficultyOption = "act_change_difficulty_option"
local actChangeHalfLengthOption = "act_change_half_length_option"

local CAMERA_OPTION_INDEX = 6
local AUTO_SWITCH_OPTION_INDEX = 1
local AUTO_SWITCH_MOVE_ASSIST_OPTION_INDEX = 2
local MOVE_ASSISTANCE_OPTION_INDEX = 3
local LARGE_BUTTONS_OPTION_INDEX = 5
local LARGE_DPAD_OPTION_INDEX = 6
local DIFFICULTY_OPTION_INDEX = 5
local HALF_LENGTH_OPTION_INDEX = 4

function customization:new(init)
  local o = init or {}
  setmetatable(o, self)
  self.__index = self
  o.services = {
    FifaCustomizationService = o.api("FifaCustomizationService"),
    settingsService = o.api("SettingsService"),
    gameStateService = o.api("GameStateService"),
    PauseMenuService = o.api("PauseMenuService")
  }
  o:initValues()
  
  o.im.RegisterAction(actChangeCameraOption, function(actionName, data)
    o:onCameraOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeAutoSwitchOption, function(actionName, data)
    o:onAutoSwitchOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeAutoSwitchMoveAssistOption, function(actionName, data)
    o:onAutoSwitchMoveAssistOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeMoveAssistanceOption, function(actionName, data)
    o:onMoveAssistanceOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeLargeButtonsOption, function(actionName, data)
    o:onLargeButtonsOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeLargeDpadOption, function(actionName, data)
    o:onLargeDpadOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeDifficultyOption, function(actionName, data)
    o:onDifficultyOptionChanged(data)
  end)
  o.im.RegisterAction(actChangeHalfLengthOption, function(actionName, data)
    o:onHalfLengthOptionChanged(data)
  end)
  
  ----------------------------------------------------------------------------------------
  o.im.Subscribe(BND_MATCH_SETTINGS_LIST_DATA, function()
    o:publishMatchSettingsListData()
  end)
  ----------------------------------------------------------------------------------------
  
  o.im.Subscribe(bndCameraOptions, function()
    o:publishCameraOptions()
  end)
  o.im.Subscribe(bndAutoSwitchOptions, function()
    o:publishAutoSwitchOptions()
  end)
  o.im.Subscribe(bndAutoSwitchMoveAssistOptions, function()
    o:publishAutoSwitchMoveAssistOptions()
  end)
  o.im.Subscribe(bndMoveAssistanceOptions, function()
    o:publishMoveAssistanceOptions()
  end)
  o.im.Subscribe(bndLargeButtonsOptions, function()
    o:publishLargeButtonsOptions()
  end)
  o.im.Subscribe(bndLargeDpadOptions, function()
    o:publishLargeDpadOptions()
  end)
  o.im.Subscribe(bndDifficultyOptions, function()
    o:publishDifficultyOptions()
  end)
  o.im.Subscribe(bndHalfLengthOptions, function()
    o:publishHalfLengthOptions()
  end)

  return o
end

function customization:initValues()
  local controllerId = self.services.gameStateService.GetPreferedControllerId()
  self.customizationOptions = self.services.settingsService.GetCustomizationOptions(controllerId)
  self:initCameraData(self.customizationOptions[CAMERA_OPTION_INDEX].data)
  self:initAutoSwitchData(self.customizationOptions[AUTO_SWITCH_OPTION_INDEX].data)
  self:initAutoSwitchMoveAssistData(self.customizationOptions[AUTO_SWITCH_MOVE_ASSIST_OPTION_INDEX].data)
  self:initMoveAssistanceData(self.customizationOptions[MOVE_ASSISTANCE_OPTION_INDEX].data)
  self.touchOptions = self.services.FifaCustomizationService.GetTouchOptions(controllerId)
  self:initLargeButtonsData(self.touchOptions[LARGE_BUTTONS_OPTION_INDEX].data)
  self:initLargeDpadData(self.touchOptions[LARGE_DPAD_OPTION_INDEX].data)
  self:initDifficultyData(self.customizationOptions[DIFFICULTY_OPTION_INDEX].data)
  self:initHalfLengthData(self.customizationOptions[HALF_LENGTH_OPTION_INDEX].data)
end

function customization:initCameraData(cameraData)
  self.cameraData = cameraData
  self.cameraData.rowChangeAction = "act_change_camera_option"
  self:setSelectedIndex(self.cameraData)
end
function customization:initAutoSwitchData(autoSwitchData)
  self.autoSwitchData = autoSwitchData
  self.autoSwitchData.rowChangeAction = "act_change_auto_switch_option"
  self:setSelectedIndex(self.autoSwitchData)
end
function customization:initAutoSwitchMoveAssistData(autoSwitchMoveAssistData)
  self.autoSwitchMoveAssistData = autoSwitchMoveAssistData
  self.autoSwitchMoveAssistData.rowChangeAction = "act_change_auto_switch_move_assist_option"
  self:setSelectedIndex(self.autoSwitchMoveAssistData)
end
function customization:initMoveAssistanceData(moveAssistanceData)
  self.moveAssistanceData = moveAssistanceData
  self.moveAssistanceData.currentValue = self:SettingToBoolean(self.moveAssistanceData.currentValue)
  self.moveAssistanceData.rowChangeAction = "act_change_move_assistance_option"
  if self.services.PauseMenuService.IsOnlineGame() then
    self.moveAssistanceData.isEnabled = false
  else
    self.moveAssistanceData.isEnabled = true
  end
  self:setSelectedIndex(self.moveAssistanceData)
end
function customization:initLargeButtonsData(largeButtonsData)
  self.largeButtonsData = largeButtonsData
  self.largeButtonsData.currentValue = self:SettingToBoolean(self.largeButtonsData.currentValue)
  self.largeButtonsData.rowChangeAction = "act_change_large_buttons_option"
  self:setSelectedIndex(self.largeButtonsData)
end
function customization:initLargeDpadData(largeDpadData)
  self.largeDpadData = largeDpadData
  self.largeDpadData.currentValue = self:SettingToBoolean(self.largeDpadData.currentValue)
  self.largeDpadData.rowChangeAction = "act_change_large_dpad_option"
  self:setSelectedIndex(self.largeDpadData)
end
function customization:initDifficultyData(difficultyData)
  self.difficultyData = difficultyData
  self.difficultyData.rowChangeAction = "act_change_difficulty_option"
  self:setSelectedIndex(self.difficultyData)
end
function customization:initHalfLengthData(halfLengthData)
  self.halfLengthData = halfLengthData
  self.halfLengthData.rowChangeAction = "act_change_half_length_option"
  self:setSelectedIndex(self.halfLengthData)
end

function customization:setSelectedIndex(optionData)
  local selectedIndex = self:GetValueIndex(optionData.data, optionData.currentValue)
  optionData.index = selectedIndex
end

function customization:publishCameraOptions()
  self.im.Publish(bndCameraOptions, self.cameraData)
end
function customization:publishAutoSwitchOptions()
  self.im.Publish(bndAutoSwitchOptions, self.autoSwitchData)
end
function customization:publishAutoSwitchMoveAssistOptions()
  self.im.Publish(bndAutoSwitchMoveAssistOptions, self.autoSwitchMoveAssistData)
end
function customization:publishMoveAssistanceOptions()
  self.im.Publish(bndMoveAssistanceOptions, self.moveAssistanceData)
end
function customization:publishLargeButtonsOptions()
  self.im.Publish(bndLargeButtonsOptions, self.largeButtonsData)
end
function customization:publishLargeDpadOptions()
  self.im.Publish(bndLargeDpadOptions, self.largeDpadData)
end
function customization:publishDifficultyOptions()
  self.im.Publish(bndDifficultyOptions, self.difficultyData)
end
function customization:publishHalfLengthOptions()
  self.im.Publish(bndHalfLengthOptions, self.halfLengthData)
end

function customization:publishRightSideLabel()
  self.im.Publish(bndRightSideLabel, self.label)
end
function customization:publishRightSideImage()
  self.im.Publish(bndRightSideImage, {
    name = self.imagetype,
    id = self.imageID
  })
end

function customization:BooleanToSetting(value)
  local setting = 1
  if value then
    setting = 0
  end
  return setting
end
function customization:SettingToBoolean(value)
  local boolean = true
  if value == 1 then
    boolean = false
  end
  return boolean
end

function customization:onCameraOptionChanged(newOption)
  local newValue = newOption.value
  if self.cameraData.currentValue ~= newValue then
    self.cameraData.currentValue = newValue
    self.services.settingsService.SaveCameraValue(newValue)
  end
end
function customization:onAutoSwitchOptionChanged(newOption)
  local newValue = newOption.value
  if self.autoSwitchData.currentValue ~= newValue then
    self.autoSwitchData.currentValue = newValue
    local controllerId = self.services.gameStateService.GetPreferedControllerId()
    self.services.settingsService.SaveAutoSwitchValue(controllerId, newValue)
    local dirtyIndices = {}
    table.insert(dirtyIndices, 1)
    if self.autoSwitchData.currentValue == 0 then
      self.autoSwitchMoveAssistData.isEnabled = false
      self:publishMatchSettingsListData(dirtyIndices)
    else
      self.autoSwitchMoveAssistData.isEnabled = true
      self:publishMatchSettingsListData(dirtyIndices)
    end
  end
end
function customization:onAutoSwitchMoveAssistOptionChanged(newOption)
  local newValue = newOption.value
  if self.autoSwitchMoveAssistData.currentValue ~= newValue then
    self.autoSwitchMoveAssistData.currentValue = newValue
    local controllerId = self.services.gameStateService.GetPreferedControllerId()
    self.services.settingsService.SaveAutoSwichMoveAssistanceValue(controllerId, newValue)
  end
end
function customization:onMoveAssistanceOptionChanged(newOption)
  local newValue = newOption.value
  if self.moveAssistanceData.currentValue ~= newValue then
    self.moveAssistanceData.currentValue = newValue
    local controllerId = self.services.gameStateService.GetPreferedControllerId()
    self.services.settingsService.SaveMoveAssistanceValue(controllerId, self:BooleanToSetting(newValue))
  end
end
function customization:onLargeButtonsOptionChanged(newOption)
  local newValue = newOption.value
  if self.largeButtonsData.currentValue ~= newValue then
    self.largeButtonsData.currentValue = newValue
    local controllerId = self.services.gameStateService.GetPreferedControllerId()
    self.services.FifaCustomizationService.SaveLargeButtonsValue(controllerId, self:BooleanToSetting(newValue))
  end
end
function customization:onLargeDpadOptionChanged(newOption)
  local newValue = newOption.value
  if self.largeDpadData.currentValue ~= newValue then
    self.largeDpadData.currentValue = newValue
    local controllerId = self.services.gameStateService.GetPreferedControllerId()
    self.services.FifaCustomizationService.SaveLargeDpadValue(controllerId, self:BooleanToSetting(newValue))
  end
end
function customization:onDifficultyOptionChanged(newOption)
  local newValue = newOption.value
  if self.difficultyData.currentValue ~= newValue then
    self.difficultyData.currentValue = newValue
    self.services.settingsService.SaveDifficultyValue(newValue)
  end
end
function customization:onHalfLengthOptionChanged(newOption)
  local newValue = newOption.value
  if self.halfLengthData.currentValue ~= newValue then
    self.halfLengthData.currentValue = newValue
    self.services.settingsService.SaveHalfLengthValue(newValue)
  end
end

function customization:publishMatchSettingsListData(pDirtyIndices)
  local dirtyIndices = {}
  local nRows = 8
  if pDirtyIndices == nil or #pDirtyIndices == 0 then
    for i = 0, nRows do
      table.insert(dirtyIndices, i)
    end
  else
    dirtyIndices = pDirtyIndices
  end
  local dataList = {
    dirty = dirtyIndices,
    data = {
      {
        data = self.cameraData
      },
      {
        data = self.autoSwitchData
      },
      {
        data = self.autoSwitchMoveAssistData
      },
      {
        data = self.moveAssistanceData
      },
      {
        data = self.largeButtonsData
      },
      {
        data = self.largeDpadData
      },
      {
        data = self.difficultyData
      },
      {
        data = self.halfLengthData
      }
    }
  }
  self.autoSwitchMoveAssistData.isEnabled = self.autoSwitchData.currentValue ~= 0
  self.im.Publish(BND_MATCH_SETTINGS_LIST_DATA, dataList)
end

function customization:GetValueIndex(data, value)
  local i = 1
  while i <= #data and data[i].value ~= value do
    i = i + 1
  end
  return i - 1
end

function customization:finalize()

  self.im.Unsubscribe(BND_MATCH_SETTINGS_LIST_DATA)
  
  self.im.Unsubscribe(bndCameraOptions)
  self.im.Unsubscribe(bndAutoSwitchOptions)
  self.im.Unsubscribe(bndAutoSwitchMoveAssistOptions)
  self.im.Unsubscribe(bndMoveAssistanceOptions)
  self.im.Unsubscribe(bndLargeButtonsOptions)
  self.im.Unsubscribe(bndLargeDpadOptions)
  self.im.Unsubscribe(bndDifficultyOptions)
  self.im.Unsubscribe(bndHalfLengthOptions)
  
  self.im.UnregisterAction(actChangeCameraOption)
  self.im.UnregisterAction(actChangeAutoSwitchOption)
  self.im.UnregisterAction(actChangeAutoSwitchMoveAssistOption)
  self.im.UnregisterAction(actChangeMoveAssistanceOption)
  self.im.UnregisterAction(actChangeLargeButtonsOption)
  self.im.UnregisterAction(actChangeLargeDpadOption)
  self.im.UnregisterAction(actChangeDifficultyOption)
  self.im.UnregisterAction(actChangeHalfLengthOption)
  
end
return customization